
#include "glut-start.h"
#include "glut-content.h"
#include "glwrap.h"
#include "textures.h"

class MyContent : public IGlutContent {
  public:
    virtual void createScene();
    virtual void step( float deltaTime );
};

IGlutContent * MakeGlutContent()
{
  return new MyContent();
}

struct GroundVertex {
  float x, y, z;
  float nx, ny, nz;
  float u, v;
};

struct FirePosition {
  float x, y, z;
  float dx, dy, dz;
};
struct FireVertex {
  float x, y, z;
  float u, v;
  unsigned int c;
};

static float const W0 = 1.0f / 16;
static float const W1 = 2.0f / 16;
static float const W2 = 4.0f / 16;
static float const AMPLITUDE = 2.0f;
static int const PASSCOUNT = 5;

static float const CONSTANT = 1;

static float intensityPhase = 0;

static GroundVertex ground[ 25 ][ 25 ];
static unsigned short gindex[ 24 ][ 24 ][ 6 ];
GWState & groundTexture() {
  static GWState state;
  return state;
}

enum {
  NUM_PARTICLES = 128
};
static FirePosition fireParticles[NUM_PARTICLES];
static unsigned int nextParticle = 0;
static float const PARTICLE_TIME = 0.05f;
// seed the system with 10 particles first time
static float timeToNextParticle = -PARTICLE_TIME * 10;
static float const PARTICLE_SIZE = 0.1f;
GWState & fireTexture() {
  static GWState state;
  return state;
}

void MyContent::createScene()
{
  srand( 5 );

  //  make a random height field
  float height[ 27 ][ 27 ], temp[ 27 ][ 27 ];
  for( int iy = 0; iy < 27; ++iy ) {
    for( int ix = 0; ix < 27; ++ix ) {
      height[iy][ix] = float(rand()) / RAND_MAX * AMPLITUDE;
    }
  }
  float minval, maxval;
  for( int pass = 0; pass < PASSCOUNT; ++pass ) {
    //  copy the edge pixels to make wrapping (for tiling)
    for( int i = 1; i < 26; ++i ) {
      height[0][i] = height[24][i];
      height[25][i] = height[1][i];
      height[26][i] = height[2][i];
      height[i][0] = height[i][24];
      height[i][25] = height[i][1];
      height[i][26] = height[i][2];
    }
    //  replicate corners
    height[0][0] = height[1][1]; height[26][0] = height[25][1]; height[0][26] = height[1][25]; height[26][26] = height[25][25];
    float tmpmin = height[1][1];
    float tmpmax = tmpmin;
    for( int iy = 1; iy < 26; ++iy ) {
      for( int ix = 1; ix < 26; ++ix ) {
        //  filter kernel
        float f = 
            (height[iy-1][ix-1] + height[iy-1][ix+1] + height[iy+1][ix-1] + height[iy+1][ix+1]) * W0 +
            (height[iy][ix-1] + height[iy][ix+1] + height[iy-1][ix] + height[iy+1][ix]) * W1 +
            height[iy][ix] * W2;
        if( f < tmpmin ) tmpmin = f;
        if( f > tmpmax ) tmpmax = f;
        temp[iy][ix] = f;
      }
    }
    memcpy( height, temp, sizeof( height ) );
    minval = tmpmin;
    maxval = tmpmax;
  }

  //  scale and offset height
  float scale = AMPLITUDE / (maxval-minval);
  float offset = -height[13][13] * scale;
  for( int iy = 0; iy < 27; ++iy ) {
    for( int ix = 0; ix < 27; ++ix ) {
      height[iy][ix] = height[iy][ix] * scale + offset;
    }
  }
  //  copy the edge pixels to make wrapping (for tiling)
  for( int i = 1; i < 26; ++i ) {
    height[0][i] = height[24][i];
    height[25][i] = height[1][i];
    height[26][i] = height[2][i];
    height[i][0] = height[i][24];
    height[i][25] = height[i][1];
    height[i][26] = height[i][2];
  }
  //  replicate corners
  height[0][0] = height[1][1]; height[26][0] = height[25][1]; height[0][26] = height[1][25]; height[26][26] = height[25][25];

  for( int iy = 0; iy < 25; ++iy ) {
    for( int ix = 0; ix < 25; ++ix ) {
      //  hoaky normal generation
      float nx = height[iy+1][ix] - height[iy+1][ix+2];
      float ny = 1;
      float nz = height[iy][ix+1] - height[iy+2][ix+1];
      float irp = 1.0f / sqrtf( nx*nx + ny*ny + nz*nz );
      ground[iy][ix].x = float(ix-13);
      ground[iy][ix].y = height[iy+1][ix+1];
      ground[iy][ix].z = float(iy-13);
      ground[iy][ix].u = ground[iy][ix].x * 0.125f;
      ground[iy][ix].v = ground[iy][ix].z * 0.125f;
      ground[iy][ix].nx = nx * irp;
      ground[iy][ix].ny = ny * irp;
      ground[iy][ix].nz = nz * irp;
    }
  }
  for( int iy = 0; iy < 24; ++iy ) {
    for( int ix = 0; ix < 24; ++ix ) {
      unsigned short base = iy * 25 + ix;
      gindex[iy][ix][0] = base;
      gindex[iy][ix][1] = base+25;
      gindex[iy][ix][2] = base+26;
      gindex[iy][ix][3] = base;
      gindex[iy][ix][4] = base+26;
      gindex[iy][ix][5] = base+1;
    }
  }

  groundTexture().texture = TextureRef( "art\\earthmoss.tga" );
  groundTexture().lit = true;
  groundTexture().blend = false;

  //  put all particles below the ground until they spawn
  for( int ix = 0; ix < NUM_PARTICLES; ++ix ) {
    fireParticles[ix].y = -10000;
  }
  fireTexture().texture = TextureRef( "art\\fireparticle.tga" );
  fireTexture().lit = false;
  fireTexture().blend = true;
  fireTexture().blendSrc = GL_ONE;
  fireTexture().blendDst = GL_ONE_MINUS_SRC_ALPHA;
  fireTexture().texMode = GL_MODULATE;
  fireTexture().test = false;
}

void MyContent::step( float deltaTime )
{
  //  step the lighting; make it flicker
  if( deltaTime > 0.1f ) {
    deltaTime = 0.1f;
  }
  intensityPhase += deltaTime * 3;
  if( intensityPhase > 3.14159265358979 ) {
    intensityPhase -= float( float(rand()) / RAND_MAX * 2 * 3.14159265358979 );
  }
  float intensity = sinf( intensityPhase ) * 0.2f + 0.8f;

  //  step the particle system
  assert( !(NUM_PARTICLES & (NUM_PARTICLES-1)) ); // must be power of two
  //  create new particles if necessary
  timeToNextParticle -= deltaTime;
  while( timeToNextParticle <= 0 ) {
    FirePosition * fp = &fireParticles[nextParticle];
    nextParticle = (nextParticle+1) & (NUM_PARTICLES-1);
    timeToNextParticle += PARTICLE_TIME;
    fp->x = 0; fp->y = 0; fp->z = 0;
    fp->dx = (float(rand()) / RAND_MAX - 0.5f) * 0.20f;
    fp->dz = (float(rand()) / RAND_MAX - 0.5f) * 0.20f;
    fp->dy = (0.4f - fabsf(fp->dx) - fabsf(fp->dy)) * 0.3f;
  }
  // move the current particles, and apply heat drift
  for( int ix = 0; ix < NUM_PARTICLES; ++ix ) {
    FirePosition * fp = &fireParticles[ix];
    fp->x += fp->dx * deltaTime;
    fp->y += fp->dy * deltaTime;
    fp->z += fp->dz * deltaTime;
    fp->dy += deltaTime * 0.1f;
    fp->dx *= 0.999f;
    fp->dz *= 0.999f;
  }


  assert( !glGetError() );
  //  set up scene lighting (the fire is a point light)
  glEnable( GL_LIGHT0 );
  assert( !glGetError() );
  static float const ambient[4] = { 0, 0, 0, 0 };
  float const diffuse[4] = { 0.7f * intensity * CONSTANT, 0.4f * intensity * CONSTANT, 0.2f * intensity * CONSTANT, 0 };
  float const specular[4] = { 0.5f * intensity * CONSTANT, 0.5f * intensity * CONSTANT, 0.1f * intensity * CONSTANT, 0 };
  glLightfv( GL_LIGHT0, GL_AMBIENT, ambient );
  glLightfv( GL_LIGHT0, GL_DIFFUSE, diffuse );
  glLightfv( GL_LIGHT0, GL_SPECULAR, specular );
  static float const pos[4] = { 0, 1, 0, 1 };
  glLightfv( GL_LIGHT0, GL_POSITION, pos );
  glLightf( GL_LIGHT0, GL_CONSTANT_ATTENUATION, CONSTANT );
  glLightf( GL_LIGHT0, GL_LINEAR_ATTENUATION, 0.35f );
  glLightf( GL_LIGHT0, GL_QUADRATIC_ATTENUATION, 0.01f );
  glEnable( GL_COLOR_MATERIAL );
  assert( !glGetError() );

  glColorMaterial( GL_FRONT, GL_AMBIENT_AND_DIFFUSE );
  static float const mtlspecular[4] = { 0.2f, 0.2f, 0.2f, 0 };
  glMaterialfv( GL_FRONT, GL_SPECULAR, mtlspecular );
  glMaterialf( GL_FRONT, GL_SHININESS, 16.0f );
  assert( !glGetError() );

  static float const lmambient[4] = { 0.1f, 0.15f, 0.25f, 1.0f };
  glLightModelfv( GL_LIGHT_MODEL_AMBIENT, lmambient );
  glLightModeli( GL_LIGHT_MODEL_COLOR_CONTROL, GL_SEPARATE_SPECULAR_COLOR );
  glLightModeli( GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE );
  assert( !glGetError() );

  //  render the ground -- 3x3 copies of the tile
  gwSetState( &groundTexture() );

  glEnableClientState( GL_VERTEX_ARRAY );
  glEnableClientState( GL_NORMAL_ARRAY );
  glEnableClientState( GL_TEXTURE_COORD_ARRAY );
  glVertexPointer( 3, GL_FLOAT, sizeof( GroundVertex ), &ground[0][0].x );
  glNormalPointer( GL_FLOAT, sizeof( GroundVertex ), &ground[0][0].nx );
  glTexCoordPointer( 2, GL_FLOAT, sizeof( GroundVertex ), &ground[0][0].u );
  assert( !glGetError() );
  for( int iy = -1; iy < 2; ++iy ) {
    for( int ix = -1; ix < 2; ++ix ) {
      glPushMatrix();
      glTranslatef( float(24*ix), 0, float(24*iy) );
      glDrawElements( GL_TRIANGLES, sizeof(gindex)/sizeof(unsigned short), GL_UNSIGNED_SHORT, gindex );
      glPopMatrix();
    }
  }
  assert( !glGetError() );

  //  render the fire
  gwSetState( &fireTexture() );
  //  generate vertices from particles
  FireVertex vertices[NUM_PARTICLES*4];
  float mvm[16];
  glGetFloatv( GL_MODELVIEW_MATRIX, mvm );
  FireVertex * fv = vertices;
  FirePosition * fp = fireParticles;
  for( int ix = 0; ix < NUM_PARTICLES; ++ix ) {
    //  age the texture offset based on current index
    float fage = ((nextParticle + NUM_PARTICLES - ix - 1) & (NUM_PARTICLES-1)) * 8 / float( NUM_PARTICLES );
    int age = (int)floor( fage );
    float u = age / 8.0f;
    float u1 = 0.5f/256;
    float u0 = 0.5f/32;

    unsigned int c = 0xffffffff;
    if( age == 7 ) {
      c = unsigned int( 0xff * (8-fage) );
      c = (c<<24) | (0xffffff);
    }
    fv->x = fp->x + mvm[0]*PARTICLE_SIZE*fage;
    fv->y = fp->y + mvm[4]*PARTICLE_SIZE*fage;
    fv->z = fp->z + mvm[8]*PARTICLE_SIZE*fage;
    fv->u = u+u1;
    fv->v = u0;
    fv->c = c;
    ++fv;

    fv->x = fp->x + mvm[1]*PARTICLE_SIZE*fage;
    fv->y = fp->y + mvm[5]*PARTICLE_SIZE*fage;
    fv->z = fp->z + mvm[9]*PARTICLE_SIZE*fage;
    fv->u = u+u1;
    fv->v = 1-u0;
    fv->c = c;
    ++fv;

    fv->x = fp->x - mvm[0]*PARTICLE_SIZE*fage;
    fv->y = fp->y - mvm[4]*PARTICLE_SIZE*fage;
    fv->z = fp->z - mvm[8]*PARTICLE_SIZE*fage;
    fv->u = u+0.125f-u1;
    fv->v = 1-u0;
    fv->c = c;
    ++fv;

    fv->x = fp->x - mvm[1]*PARTICLE_SIZE*fage;
    fv->y = fp->y - mvm[5]*PARTICLE_SIZE*fage;
    fv->z = fp->z - mvm[9]*PARTICLE_SIZE*fage;
    fv->u = u+0.125f-u1;
    fv->v = u0;
    fv->c = c;
    ++fv;

    ++fp;
  }
  glDisableClientState( GL_NORMAL_ARRAY );
  glEnableClientState( GL_COLOR_ARRAY );
  glVertexPointer( 3, GL_FLOAT, sizeof( FireVertex ), &vertices[0].x );
  glTexCoordPointer( 2, GL_FLOAT, sizeof( FireVertex ), &vertices[0].u );
  glColorPointer( 4, GL_UNSIGNED_BYTE, sizeof( FireVertex ), &vertices[0].c );
  glDrawArrays( GL_QUADS, 0, NUM_PARTICLES * 4 );
  glDisableClientState( GL_COLOR_ARRAY );
}
