
#include "glut-start.h"

#include <GL/GL.h>

#include "glwrap.h"
#include "textures.h"
#include "mytga.h"
#include "myext.h"
#include "internaltexture.h"



#if !defined( NDEBUG )
#define TexRefCountMax 20
#else
#define TexRefCountMax 1000
#endif


static InternalTexture sNullTexture;


static bool getTgaInfo( char * data, long size, int * width, int * height, int * bits )
{
  TGA_header const * hdr = (TGA_header *)data;
  if( hdr->imageIDLengthInBytes != 0 ) {
    warning( "TGA image ID is not supported (length %d, or not a TGA)\n", hdr->imageIDLengthInBytes );
    return false;
  }
  if( hdr->colorMapType != TGA_colorMapType_direct ) {
    warning( "colorMapType %d is not TGA direct\n", hdr->colorMapType );
    return false;
  }
  if( hdr->imageTypeCode != TGA_imageTypeCode_direct ) {
    warning( "imageTypeCode %d is not TGA direct\n", hdr->imageTypeCode );
    return false;
  }
  if( hdr->imagePixelSize != 32 ) {
    warning( "imagePixelSize %d is not TGA 32 bits\n", hdr->imagePixelSize );
    return false;
  }
  if( (hdr->imageDescriptor & TGA_imageDescriptor_alphaBitCount) != 8 ) {
    warning( "imageDescriptor %d is not TGA 8 bits alpha\n", hdr->imageDescriptor );
    return false;
  }
  *width = hdr->widthOfImage;
  *height = hdr->heightOfImage;
  *bits = hdr->imagePixelSize;
  if( *width < 1 || *width > 1024 || *height < 1 || *height > 1024 ) {
    warning( "image dimensions %dx%d are not within acceptable TGA limits\n", *width, *height );
    return false;
  }
  return true;
}



class TextureManager
{
public:
  TextureManager( );
  ~TextureManager( );

  InternalTexture *   getTexture( char const * path );
  InternalTexture *   getTexture( InternalTexture * t );
  void                putTexture( InternalTexture * t );

private:
  InternalTexture *   findOrMakeEntry( char const * path );
  InternalTexture *   recursiveFindOrMakeEntry( InternalTexture ** root, char const * path );
  void                readAndUploadEntry( InternalTexture * t );
  void                removeEntry( InternalTexture * t );
  void                recursiveDelete( InternalTexture * t );

  InternalTexture *   mRoot;
};


TextureManager::TextureManager( )
{
  mRoot = 0;
}

TextureManager::~TextureManager( )
{
  recursiveDelete( mRoot );
  mRoot = 0;
}

InternalTexture *   
TextureManager::getTexture( char const * path )
{
  if( !path || !path[0] ) return &sNullTexture;
  assert( strlen( path ) < PathLengthMax );
  if( strlen( path ) > PathLengthMax ) return 0;
  InternalTexture * t = findOrMakeEntry( path );
  assert( t->mRefCount < TexRefCountMax );
  t->mRefCount += 1;
  return t;
}

InternalTexture *   
TextureManager::getTexture( InternalTexture * t )
{
  t->mRefCount += 1;
  return t;
}

void   
TextureManager::putTexture( InternalTexture * t )
{
  if( t == &sNullTexture || !mRoot ) {    //  we allow NULL mRoot because of shutdown case
    return;
  }
  assert( t->mRefCount > 0 );
  t->mRefCount -= 1;
  if( t->mRefCount == 0 ) {
    if( t->mTexId ) {
      //  I might hold on to the texture image here, 
      //  as the driver should do a decent job of managing 
      //  the active working set. Oh, well.
      glDeleteTextures( 1, &t->mTexId );
      t->mTexId = 0;
    }
    //  I don't ever delete/remove file name records,
    //  as they're likely to be needed again some day.
  }
}

InternalTexture *   
TextureManager::findOrMakeEntry( char const * path )
{
  InternalTexture * t = recursiveFindOrMakeEntry( &mRoot, path );
  if( !t->mTexId ) {
    readAndUploadEntry( t );
  }
  return t;
}

InternalTexture *
TextureManager::recursiveFindOrMakeEntry( InternalTexture ** root, char const * path )
{
  if( !*root ) {
    InternalTexture * ret = new InternalTexture;
    strncpy( ret->mPath, path, PathLengthMax );
    ret->mPath[ PathLengthMax-1 ] = 0;
    *root = ret;
    return ret;
  }
  int cmp = stricmp( (*root)->mPath, path );
  if( !cmp ) {
    return *root;
  }
  if( cmp < 0 ) {
    return recursiveFindOrMakeEntry( &(*root)->mLeft, path );
  }
  return recursiveFindOrMakeEntry( &(*root)->mRight, path );
}

void                
TextureManager::readAndUploadEntry( InternalTexture * t )
{
  FILE * f = fopen( t->mPath, "rb" );
  if( !f ) {
    warning( "texture %s doesn't exist\n", t->mPath );
    return;
  }
  fseek( f, 0, 2 );
  long l = ftell( f );
  StackScratch s;
  void * space = s.get( l );
  if( !space ) {
    error( "texture %s seems to be too large (is %ld, we supposedly support %ld)\n", t->mPath, l, ScratchSizeMax );
    fclose( f );
    return;  
  }
  fseek( f, 0, 0 );
  fread( space, l, 1, f );
  fclose( f );
  int width = 0, height = 0, bits = 0;
  if( !getTgaInfo( (char *)space, l, &width, &height, &bits ) ) {
    warning( "texture %s is not a TGA\n", t->mPath );
    return;
  }
  if( (width & (width-1)) || (height & (height-1)) || (bits & (bits-1)) ) {
    warning( "texture %s is non-power-of-two size %dx%dx%d\n", t->mPath, width, height, bits );
    return;
  }
  else {
    debug( "texture %s is sized %dx%dx%d\n", t->mPath, width, height, bits );
  }
  glGenTextures( 1, &t->mTexId );
  glBindTexture( GL_TEXTURE_2D, t->mTexId );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
  assert( bits == 32 ); //fixme:  for now
  glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_BGRA, GL_UNSIGNED_BYTE, NULL );
  glTexSubImage2D( GL_TEXTURE_2D, 0, 0, 0, width, height, GL_BGRA, GL_UNSIGNED_BYTE, ((char*)space)+18 );
  gwFlushState( );

  t->mWidth = width;
  t->mHeight = height;

  assert( !glGetError( ) );
}

void
TextureManager::recursiveDelete( InternalTexture * t )
{
  if( !t ) return;
  recursiveDelete( t->mLeft );
  recursiveDelete( t->mRight );
  glDeleteTextures( 1, &t->mTexId );
  delete t;
}


static TextureManager sTexManager;



TextureRef TextureRef::null;

TextureRef::TextureRef( )
{
  mTexture = sTexManager.getTexture( &sNullTexture );
}

TextureRef::TextureRef( char const * path )
{
  mTexture = sTexManager.getTexture( path );
}

TextureRef::TextureRef( TextureRef const & other )
{
  mTexture = sTexManager.getTexture( other.mTexture );
}

TextureRef::~TextureRef( )
{
  sTexManager.putTexture( mTexture );
  mTexture = 0;
}

TextureRef &
TextureRef::operator=( TextureRef const & other )
{
  InternalTexture * temp = sTexManager.getTexture( other.mTexture );
  sTexManager.putTexture( mTexture );
  mTexture = temp;
  return *this;
}

bool 
TextureRef::operator<( TextureRef const & other ) const
{
  return stricmp( mTexture->mPath, other.mTexture->mPath ) < 0;
}

bool 
TextureRef::consistent( ) const
{
  return mTexture->consistent( );
}

void 
TextureRef::getSize( int * xSize, int * ySize )
{
  *xSize = mTexture->mWidth;
  *ySize = mTexture->mHeight;
}

void 
TextureRef::getInset( float * xInset, float * yInset )
{
  *xInset = 1.f/(2*mTexture->mWidth);
  *yInset = 1.f/(2*mTexture->mHeight);
}

